/* -*- Mode: Vala; indent-tabs-mode: nil; tab-width: 2 -*-
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 * SPDX-FileCopyrightText: Michael Terry
 */

using GLib;

/* This is meant to be used right after a successful OperationBackup to
   verify the results. */

public class DejaDup.OperationVerify : Operation
{
  public string tag {get; construct;}

  public OperationVerify(Backend backend, string tag) {
    // Should we nag user about password, etc?  What this really means is that
    // we try to do our normal verification routine in as close an emulation
    // to a fresh restore after a disaster as possible.  So fresh cache, no
    // saved password, etc.  We do *not* explicitly unmount the backend,
    // because we may not be the only consumers.
    var mode = is_nag_time() ? ToolJob.Mode.VERIFY_CLEAN : ToolJob.Mode.VERIFY_BASIC;

    Object(mode: mode, backend: backend, tag: tag);
    if (mode == ToolJob.Mode.VERIFY_CLEAN) {
      use_cached_password = false;
    }
  }

  public async override void start()
  {
    if (mode == ToolJob.Mode.VERIFY_CLEAN) {
      var tmp_state = get_state();
      tmp_state.passphrase = null;
      set_state(tmp_state);
    }
    yield base.start();
  }

  protected override void connect_to_job(bool actions = true)
  {
    job.tag = tag;
    // Don't connect to action_desc because we don't want Duplicity's
    // "Restoring..." to confuse people doing a backup.
    base.connect_to_job(false /* actions */);
  }

  internal async override void operation_finished(bool success, bool cancelled)
  {
    if (success && !cancelled && mode == ToolJob.Mode.VERIFY_CLEAN) {
      update_nag_time();
    }

    yield base.operation_finished(success, cancelled);
  }
}
