package org.opentest4j.reporting.tooling.spi.htmlreport;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link Image}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code new Image.Builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableImage implements Image {
  private final String content;
  private final String altText;

  private ImmutableImage(String content, String altText) {
    this.content = content;
    this.altText = altText;
  }

  /**
   * @return The value of the {@code content} attribute
   */
  @Override
  public String getContent() {
    return content;
  }

  /**
   * @return The value of the {@code altText} attribute
   */
  @Override
  public String getAltText() {
    return altText;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Image#getContent() content} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for content
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableImage withContent(String value) {
    String newValue = Objects.requireNonNull(value, "content");
    if (this.content.equals(newValue)) return this;
    return new ImmutableImage(newValue, this.altText);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Image#getAltText() altText} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for altText
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableImage withAltText(String value) {
    String newValue = Objects.requireNonNull(value, "altText");
    if (this.altText.equals(newValue)) return this;
    return new ImmutableImage(this.content, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableImage} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableImage
        && equalsByValue((ImmutableImage) another);
  }

  private boolean equalsByValue(ImmutableImage another) {
    return content.equals(another.content)
        && altText.equals(another.altText);
  }

  /**
   * Computes a hash code from attributes: {@code content}, {@code altText}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + content.hashCode();
    h += (h << 5) + altText.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Image} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Image{"
        + "content=" + content
        + ", altText=" + altText
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Image} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Image instance
   */
  public static Image copyOf(Image instance) {
    if (instance instanceof ImmutableImage) {
      return (ImmutableImage) instance;
    }
    return new Image.Builder()
        .content(instance.getContent())
        .altText(instance.getAltText())
        .build();
  }

  /**
   * Builds instances of type {@link Image Image}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static class Builder {
    private static final long INIT_BIT_CONTENT = 0x1L;
    private static final long INIT_BIT_ALT_TEXT = 0x2L;
    private long initBits = 0x3L;

    private String content;
    private String altText;

    /**
     * Creates a builder for {@link Image Image} instances.
     * <pre>
     * new Image.Builder()
     *    .content(String) // required {@link Image#getContent() content}
     *    .altText(String) // required {@link Image#getAltText() altText}
     *    .build();
     * </pre>
     */
    public Builder() {
      if (!(this instanceof Image.Builder)) {
        throw new UnsupportedOperationException("Use: new Image.Builder()");
      }
    }

    /**
     * Initializes the value for the {@link Image#getContent() content} attribute.
     * @param content The value for content 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Image.Builder content(String content) {
      this.content = Objects.requireNonNull(content, "content");
      initBits &= ~INIT_BIT_CONTENT;
      return (Image.Builder) this;
    }

    /**
     * Initializes the value for the {@link Image#getAltText() altText} attribute.
     * @param altText The value for altText 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Image.Builder altText(String altText) {
      this.altText = Objects.requireNonNull(altText, "altText");
      initBits &= ~INIT_BIT_ALT_TEXT;
      return (Image.Builder) this;
    }

    /**
     * Builds a new {@link Image Image}.
     * @return An immutable instance of Image
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public Image build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableImage(content, altText);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_CONTENT) != 0) attributes.add("content");
      if ((initBits & INIT_BIT_ALT_TEXT) != 0) attributes.add("altText");
      return "Cannot build Image, some of required attributes are not set " + attributes;
    }
  }
}
