#!/usr/bin/env bats
# Test a realm with current and legacy enctypes

. ./krb5/util

setup() {
    load '/usr/lib/bats/bats-assert/load'
    load '/usr/lib/bats/bats-support/load' # this is required by bats-assert!
}

grep_enc_types() {
    grep -Po 'Key:\s*vno\s*\d+,\s*\K[^ ]+' | paste -sd ' ' -
}

## This could be a setup_file, but that seems to
## create pre-auth issues when asking for the ticket
test_realm_with_enctypes() {
    local enctypes=$1

    TEST_REALM="EXAMPLE.INTERNAL"
    MYHOSTNAME="krb5-dep8.internal"
    adjust_hostname "${MYHOSTNAME}"

    create_realm_enctypes "${TEST_REALM}" "${MYHOSTNAME}" "${enctypes}"

    # create a random-enough principal
    principal="testuser$$"
    kadmin.local -q "addprinc -pw secret ${principal}"

    # print enctypes for principal to TAP stream
    princ_info=$(kadmin.local -q "getprinc ${principal}")
    echo "# enctypes = $(echo ${princ_info} | grep_enc_types)" >&3

    # get a ticket
    echo secret | kinit ${principal}

    # did we really get a ticket?
    klist | grep krbtgt/${TEST_REALM}@${TEST_REALM}

    # destroy it
    kdestroy

    # delete the principal
    kadmin.local -q "delprinc -force ${principal}"
}

kinit_cred_error="kinit: Generic error (see e-text) while getting initial credentials"
# If the ticket was granted, the end of the transaction gives this message
success_kinit_msg="Make sure that you have removed this principal from all ACLs before reusing."
default_enctypes="aes256-cts-hmac-sha1-96 aes128-cts-hmac-sha256-128 aes256-cts-hmac-sha1-96 aes128-cts-hmac-sha1-96"

@test "LEGACY accepts default supported enctypes" {
    update-crypto-policies --set LEGACY > /dev/null
    run test_realm_with_enctypes "${default_enctypes}"
    assert_output --partial "${success_kinit_msg}"
}
@test "DEFAULT accepts default supported enctypes" {
    update-crypto-policies --set DEFAULT > /dev/null
    run test_realm_with_enctypes "${default_enctypes}"
    assert_output --partial "${success_kinit_msg}"
}
@test "FUTURE rejects default supported enctypes" {
    update-crypto-policies --set FUTURE > /dev/null
    run test_realm_with_enctypes "${default_enctypes}"
    assert_output --partial "${kinit_cred_error}"
}

strong_enctypes="aes256-cts-hmac-sha384-192"
@test "LEGACY accepts strong supported enctypes" {
    update-crypto-policies --set LEGACY > /dev/null
    run test_realm_with_enctypes "${strong_enctypes}"
    assert_output --partial "${success_kinit_msg}"
}
@test "DEFAULT accepts strong supported enctypes" {
    update-crypto-policies --set DEFAULT > /dev/null
    run test_realm_with_enctypes "${strong_enctypes}"
    assert_output --partial "${success_kinit_msg}"
}
@test "FUTURE accepts strong supported enctypes" {
    update-crypto-policies --set FUTURE > /dev/null
    run test_realm_with_enctypes "${strong_enctypes}"
    assert_output --partial "${success_kinit_msg}"
}

teardown_file() {
    # Kill KDC
    KRB5KDC_PID=$(pidof krb5kdc)
    if [[ -n "$KRB5KDC_PID" ]]; then
        kill "$KRB5KDC_PID" 2>/dev/null || true
        wait "$KRB5KDC_PID" 2>/dev/null || true
    fi
}
