// @HEADER
// *****************************************************************************
//                 Belos: Block Linear Solvers Package
//
// Copyright 2004-2016 NTESS and the Belos contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

/*! \file BelosTpetraOperator.h
    \brief This file provides a Tpetra::Operator interface so Belos can be integrated into
     other codes as an abstract operator.
*/

#ifndef BELOS_TPETRA_OPERATOR_H
#define BELOS_TPETRA_OPERATOR_H

#include "Tpetra_MultiVector.hpp"
#include "Tpetra_Operator.hpp"
#include "Tpetra_Map.hpp"

#include "BelosLinearProblem.hpp"
#include "BelosStatusTest.hpp"
#include "BelosOutputManager.hpp"
#include "BelosTpetraAdapter.hpp"
#include "BelosSolverFactory.hpp"

#include "Teuchos_ParameterList.hpp"

/*! \class Belos::TpetraOperator
    \brief This class provides an interface to the Tpetra::Operator class, so Belos can be 
    integrated into other codes as an abstract operator. It allows a Belos solver to be
    called a Tpetra::Operator.  
    Thus, it can use itself as a preconditioner if need be.  It can
    also be used as the inner iteration of Anasazi :)
*/

namespace Belos {

template<class Scalar = Tpetra::Operator<>::scalar_type,
         class LocalOrdinal = Tpetra::Operator<>::local_ordinal_type,
         class GlobalOrdinal = Tpetra::Operator<>::global_ordinal_type,
         class Node = Tpetra::Operator<>::node_type>
class TpetraOperator : public Tpetra::Operator<Scalar, LocalOrdinal, GlobalOrdinal, Node> {
private:

  // Typedefs:
  using Tpetra_MultiVector = Tpetra::MultiVector<Scalar, LocalOrdinal, GlobalOrdinal, Node>;
  using Tpetra_Operator = Tpetra::Operator<Scalar, LocalOrdinal, GlobalOrdinal, Node>;
  using Tpetra_Map = Tpetra::Map<LocalOrdinal,GlobalOrdinal,Node>;

  Teuchos::RCP<SolverManager<Scalar,Tpetra_MultiVector,Tpetra_Operator> > solver_;
  Teuchos::RCP<LinearProblem<Scalar,Tpetra_MultiVector,Tpetra_Operator> > lp_;
  Teuchos::RCP<Teuchos::ParameterList> plist_;

  //! Name of solver to be passed into solver factory. 
  std::string solver_name_;

  /// \brief True if TpetraOperator should initialize the solution vector to zero before
  ///   calling 'solve' in the 'apply' function.  
  bool initSolnVec_;

public:

  //! @name Constructor / Destructor
  //@{ 
  
    /// \brief Create a Tpetra::Operator from the Belos solver \c solver_name  generated by Belos' solver factory.
    ///
    /// The solver is initialized with a LinearProblem (\c lp) and ParameterList (\c plist).
    ///
    /// One should create a new linear problem to pass to the inner solver which is separate
    /// from the linear problem passed to any outer solver. (Both problems may have the same 
    /// operator, but solution and right-hand side vectors need to be maintained separately.)  
    ///
    /// When true, the bool \c initSolnVec causes the solution vector \c Y to be itialized to zero
    /// when apply(X,Y) is called.

  TpetraOperator( const Teuchos::RCP<LinearProblem<Scalar,Tpetra_MultiVector,Tpetra_Operator> >& lp, 
		  const Teuchos::RCP<Teuchos::ParameterList>& plist,
      const std::string solver_name,
      bool initSolnVec = false )  
  : lp_(lp), 
    plist_(plist),
    solver_name_(solver_name),
    initSolnVec_(initSolnVec)
  {
    Belos::SolverFactory<Scalar,Tpetra_MultiVector,Tpetra_Operator> factory;
    solver_ = factory.create( solver_name_, plist );
    solver_->setProblem( lp_ );
  }

  
  //! Destructor
  virtual ~TpetraOperator() = default;
  //@}
  
  //! @name Operator application methods
  //@{ 
  
  /// \brief Apply the operator.  Performs a linear solve with right-hand side \c X.
  /// Solution is stored in \c Y.  If \c initSolnVec was set to true, the vector \c Y
  /// will be initialized with zeros before calling solve.  If \c initSolnVec is false, 
  /// the vector \c Y will be used as an initial guess for the linear solve.  
  ///
  /// When using the GmresPoly solver, the polynomial will be applied to \c X at each call
  /// instead of performing a linear solve.  

  void apply( const Tpetra_MultiVector &X, Tpetra_MultiVector &Y, 
           Teuchos::ETransp mode = Teuchos::NO_TRANS,
           Scalar alpha = Teuchos::ScalarTraits<Scalar>::one(),
           Scalar beta = Teuchos::ScalarTraits<Scalar>::zero()) const override
  {
    Teuchos::RCP<const Tpetra_MultiVector> vec_X = Teuchos::rcpFromRef( X );
    Teuchos::RCP<Tpetra_MultiVector> vec_Y = Teuchos::rcpFromRef( Y );
    if (initSolnVec_)
    {
      vec_Y->putScalar( 0.0 );
      lp_->setInitResVec( vec_X );
    }
    lp_->setProblem( vec_Y, vec_X );
    solver_->solve();
  };
  //@}
  
  //! @name Attribute access functions
  //@{ 

  //! Return the name of the requested solver.
  const std::string getSolverName() const { return solver_name_; };
  
  //! Return whether the operator supports applying its transpose or conjugate transpose.
  bool hasTransposeApply() const override { return false; };
  
  //! Return the domain map for this operator.
  Teuchos::RCP<const Tpetra_Map> getDomainMap() const override
   { return lp_->getOperator()->getDomainMap(); };
  
  //! Return the range map for this operator.
  Teuchos::RCP<const Tpetra_Map> getRangeMap() const override	
   { return lp_->getOperator()->getRangeMap(); };
  //@}	   

};

} //end namespace Belos

// end of file BELOS_TPETRA_OPERATOR_H
#endif 

